const express = require('express');
const { get, all, run } = require('../db/init');
const authMiddleware = require('../middleware/auth');

const router = express.Router();

// Get all jobs (public - only published jobs, or all for admin)
router.get('/', (req, res) => {
    try {
        let query = 'SELECT * FROM jobs';
        const params = [];

        // Only show published jobs for public access
        if (!req.user) {
            query += ' WHERE is_published = 1';
        }

        query += ' ORDER BY posted_date DESC, created_at DESC';

        const jobs = all(query, params);
        res.json(jobs);
    } catch (err) {
        console.error('Error fetching jobs:', err);
        res.status(500).json({ error: 'Failed to fetch jobs' });
    }
});

// Get single job by ID (public)
router.get('/:id', (req, res) => {
    try {
        const job = get('SELECT * FROM jobs WHERE id = ?', [req.params.id]);

        if (!job) {
            return res.status(404).json({ error: 'Job not found' });
        }

        // Only show published jobs to public
        if (!req.user && !job.is_published) {
            return res.status(404).json({ error: 'Job not found' });
        }

        res.json(job);
    } catch (err) {
        console.error('Error fetching job:', err);
        res.status(500).json({ error: 'Failed to fetch job' });
    }
});

// Get job by slug (public)
router.get('/slug/:slug', (req, res) => {
    try {
        const job = get('SELECT * FROM jobs WHERE slug = ?', [req.params.slug]);

        if (!job) {
            return res.status(404).json({ error: 'Job not found' });
        }

        // Only show published jobs to public
        if (!req.user && !job.is_published) {
            return res.status(404).json({ error: 'Job not found' });
        }

        res.json(job);
    } catch (err) {
        console.error('Error fetching job:', err);
        res.status(500).json({ error: 'Failed to fetch job' });
    }
});

// Create job (protected)
router.post('/', authMiddleware, (req, res) => {
    try {
        const {
            slug,
            title,
            department,
            location,
            employment_type,
            description,
            requirements,
            responsibilities,
            salary_range,
            benefits,
            is_published,
            is_featured,
            application_deadline
        } = req.body;

        // Validate required fields
        if (!slug || !title || !description) {
            return res.status(400).json({ error: 'Missing required fields: slug, title, description' });
        }

        const result = run(`
            INSERT INTO jobs (
                slug, title, department, location, employment_type,
                description, requirements, responsibilities,
                salary_range, benefits, is_published, is_featured,
                application_deadline
            )
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
        `, [
            slug,
            title,
            department || '',
            location || '',
            employment_type || 'Full-time',
            description,
            requirements || '',
            responsibilities || '',
            salary_range || '',
            benefits || '',
            is_published ? 1 : 0,
            is_featured ? 1 : 0,
            application_deadline || null
        ]);

        res.status(201).json({
            id: result.lastInsertRowid,
            message: 'Job created successfully'
        });
    } catch (err) {
        console.error('Error creating job:', err);
        if (err.message?.includes('UNIQUE constraint failed')) {
            return res.status(400).json({ error: 'Job slug already exists' });
        }
        res.status(500).json({ error: 'Failed to create job' });
    }
});

// Update job (protected)
router.put('/:id', authMiddleware, (req, res) => {
    try {
        const {
            slug,
            title,
            department,
            location,
            employment_type,
            description,
            requirements,
            responsibilities,
            salary_range,
            benefits,
            is_published,
            is_featured,
            application_deadline
        } = req.body;

        // Validate required fields
        if (!slug || !title || !description) {
            return res.status(400).json({ error: 'Missing required fields: slug, title, description' });
        }

        run(`
            UPDATE jobs 
            SET slug = ?, title = ?, department = ?, location = ?, 
                employment_type = ?, description = ?, requirements = ?,
                responsibilities = ?, salary_range = ?, benefits = ?,
                is_published = ?, is_featured = ?, application_deadline = ?,
                updated_at = CURRENT_TIMESTAMP
            WHERE id = ?
        `, [
            slug,
            title,
            department || '',
            location || '',
            employment_type || 'Full-time',
            description,
            requirements || '',
            responsibilities || '',
            salary_range || '',
            benefits || '',
            is_published ? 1 : 0,
            is_featured ? 1 : 0,
            application_deadline || null,
            req.params.id
        ]);

        res.json({ message: 'Job updated successfully' });
    } catch (err) {
        console.error('Error updating job:', err);
        if (err.message?.includes('UNIQUE constraint failed')) {
            return res.status(400).json({ error: 'Job slug already exists' });
        }
        res.status(500).json({ error: 'Failed to update job' });
    }
});

// Delete job (protected)
router.delete('/:id', authMiddleware, (req, res) => {
    try {
        const job = get('SELECT * FROM jobs WHERE id = ?', [req.params.id]);

        if (!job) {
            return res.status(404).json({ error: 'Job not found' });
        }

        run('DELETE FROM jobs WHERE id = ?', [req.params.id]);
        res.json({ message: 'Job deleted successfully' });
    } catch (err) {
        console.error('Error deleting job:', err);
        res.status(500).json({ error: 'Failed to delete job' });
    }
});

module.exports = router;
