const initSqlJs = require('sql.js');
const fs = require('fs');
const path = require('path');

const dbPath = path.join(__dirname, 'cms.db');
const schemaPath = path.join(__dirname, 'schema.sql');

let db = null;

// Initialize database
async function initDatabase() {
    console.log('🔧 Initializing database...');

    const SQL = await initSqlJs();

    // Load existing database or create new one
    if (fs.existsSync(dbPath)) {
        const buffer = fs.readFileSync(dbPath);
        db = new SQL.Database(buffer);
        console.log('✅ Loaded existing database');
    } else {
        db = new SQL.Database();

        // Run schema
        const schema = fs.readFileSync(schemaPath, 'utf-8');
        db.run(schema);

        // Save to file
        saveDatabase();
        console.log('✅ Created new database with schema');
    }

    return db;
}

// Save database to file
function saveDatabase() {
    if (db) {
        const data = db.export();
        const buffer = Buffer.from(data);
        fs.writeFileSync(dbPath, buffer);
    }
}

// Get database instance
function getDb() {
    return db;
}

// Wrapper for db.prepare().get()
function get(sql, params = []) {
    const stmt = db.prepare(sql);
    stmt.bind(params);
    if (stmt.step()) {
        const result = stmt.getAsObject();
        stmt.free();
        return result;
    }
    stmt.free();
    return null;
}

// Wrapper for db.prepare().all()
function all(sql, params = []) {
    const stmt = db.prepare(sql);
    stmt.bind(params);
    const results = [];
    while (stmt.step()) {
        results.push(stmt.getAsObject());
    }
    stmt.free();
    return results;
}

// Wrapper for db.run()
function run(sql, params = []) {
    db.run(sql, params);
    saveDatabase();
    return {
        lastInsertRowid: db.exec("SELECT last_insert_rowid()")[0]?.values[0]?.[0] || 0,
        changes: db.getRowsModified()
    };
}

// Run if called directly
if (require.main === module) {
    initDatabase().then(() => {
        console.log('Database initialization complete!');
        process.exit(0);
    }).catch(err => {
        console.error('Error:', err);
        process.exit(1);
    });
}

module.exports = { initDatabase, getDb, get, all, run, saveDatabase };
